package gov.va.genisis2.dao.impl;

import java.util.List;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Projections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.dao.IAttachmentDAO;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.Attachment;

/**
 * @author PII
 * 
 */
/**
 * The Class AttachmentDAO.
 * 
 * The AttachmentDAO data access object (DAO) is an object that provides an
 * abstract interface to some type of database or other persistence mechanism.
 * By mapping application calls to the persistence layer, AttachmentDAO provide
 * some specific data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class AttachmentDAO implements IAttachmentDAO {

	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(AttachmentDAO.class);

	@Autowired
	private SessionFactory sessionFactory;

	@Override
	public int submitOrModify(Attachment entity) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("submitOrModify (Attachment) ");
		}

		Integer attachementId;
		if (entity.getAttachmentId() == 0) {
			attachementId = this.maxRowValueAttachment();
			entity.setAttachmentId(attachementId);
		}
		try {
			Session session = sessionFactory.getCurrentSession();
			session.saveOrUpdate(entity);
			session.flush();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify Attachment.", ex);
			throw new GenisisDAOException("Exception occurred on submitOrModify Attachment.", ex);
		}
		return entity.getAttachmentId();
	}

	@Override
	public List<Attachment> getAttachmentByRequestId(int requestId) throws GenisisDAOException {

		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getAttachmentByRequestId ");
		}

		List<Attachment> listAttachment = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Attachment> query = criteriaBuilder.createQuery(Attachment.class);
			Root<Attachment> root = query.from(Attachment.class);
			Predicate condition1 = criteriaBuilder.equal(root.get("request").get("id"), requestId);
			Predicate conditions = criteriaBuilder.and(condition1);
			query.where(conditions);
			listAttachment = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getAttachmentByRequestId.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getAttachmentByRequestId.", ex);
		}
		return listAttachment;
	}

	/**
	 * This method is used to get max value record from the table.
	 * 
	 * @return int This returns max value record plus one.
	 */
	private int maxRowValueAttachment() {
		// TODO: createCriteria Deprecated
		int maxRowVal = 1;
		Integer result = 0;
		try {
			Session session = sessionFactory.getCurrentSession();
			@SuppressWarnings("deprecation")
			Criteria criteria = session.createCriteria(Attachment.class).setProjection(Projections.max("attachmentId"));
			result = (Integer) criteria.uniqueResult();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on maxRowValue().", ex);
		}
		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		return maxRowVal;
	}

	/**
	 * @param sessionFactory
	 *            the sessionFactory to set
	 */
	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}

}
